function OUT = rungfs(input, gfs)
% ==============================================
%   PROGRAM AIM :
%         TO RUN A GFS
% ==============================================
%   HOW TO RUN :
%         1. Specify input matrix e.g. input = [10 20]
%         1. Get a FLS in structure form e.g. gfs = rdgfs('Truck.GFS')
%         2. Run as OUT = rungfs(input, gfs)
% ==============================================
%   Files Required
%         rdgfs.m
%         rungfs.m
%         Truck.GFS
% ==============================================
%   SATVIR SINGH SIDHU, ARUN KHOSLA, JASBIR SINGH SAINI
%   MAY 2009
%   COPYRIGHT RESERVED
% ==============================================

if nargin < 2
    disp('Need at least two input');
    OUT = [];
    return
end

if ~isgfs(gfs)
    error('The second argument must be a GFS structure.')
end
[M,N] = size(input);
Nin = length(gfs.Input);
if N~=Nin
    error(sprintf('%s\n%s',...
        'The first argument should have as many columns as input variables and',...
        'as many rows as independent sets of input values.'))
end
XA = 1000;
%   GFS Classifications
switch gfs.Fuzzification
    case 'Simple Type-1'
        if strcmp(gfs.Approach, 'T1 FS')
            OUT = st1gfs(input, gfs, XA);
        end
    case 'Interval Type-2'
        if strcmp(gfs.Approach, 'Left FS + Right FS')
            OUT = it21gfs(input, gfs, XA);
        elseif strcmp(gfs.Approach, 'Principal MF + FOU')
            OUT = it22gfs(input, gfs, XA);
        elseif strcmp(gfs.Approach, 'Lower MF + Upper MF')
            OUT = it23gfs(input, gfs, XA);
        end
    case 'Triangular Type-2'
        if strcmp(gfs.Approach, 'Left FS + Principal FS + Right FS')
            OUT = tt21gfs(input, gfs, XA);
        elseif strcmp(gfs.Approach, 'Base MF + FOU + Principal MF')
        elseif strcmp(gfs.Approach, 'Upper MF + Lower MF + Principal MF')
        end
end

% SIMPLE TYPE-1 FUZZY SYSTEM
% =====================
function DFZ = st1gfs(input, gfs, XA)
NumIn = length(gfs.Input);
NumOut = length(gfs.Output);
NumRule = length(gfs.Rule);
for i = 1:NumIn
    UDI(i, :) = gfs.Input(1, i).UOD;
end
for j = 1:NumOut
    UDO(j, :) = gfs.Output(1, j).UOD;
end
UD = [UDI; UDO];

for i =1:NumRule
    AMAT(i, :) = gfs.Rule(1, i).antecedent;
    CMAT(i, :) = gfs.Rule(1, i).consequent;
    OP(i) = gfs.Rule(1, i).connection;
end
MFN = [AMAT CMAT];
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        if MFN(i, j) < 0, INV(i, j) = 1;
        else, INV(i, j) = 0;
        end
    end
end

for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        U = UD(j, :);
        u = linspace(U(1), U(2), XA);
        if j <= NumIn
            if MFN(i, j) ~= 0
                S = gfs.Input(1, j).mf(1, abs(MFN(i, j))).Shape;                                        % SHAPES
                P = eval(gfs.Input(1, j).mf(1, abs(MFN(i, j))).Params);             % PARAMETER
            else
                S = 'None';
            end
        elseif j > NumIn
            if MFN(i, j) ~= 0
                S = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).Shape;
                P = eval(gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).Params);
            else
                S = 'None';
            end
        end
        switch S
            case 'None'
                F = ones(1, length(u));
            case 'Gaussian'
                F = gaussmf(u, [P(1) P(2)]);
            case 'Trapezoidal'
                F = trapmf(u, [P(1) P(2) P(3) P(4)]);
            case 'Triangular'
                F = trimf(u, [P(1) P(2) P(3)]);
            case 'S-Shaped'
                F = smf(u, [P(1) P(2)]);
            case 'Z-Shaped'
                F = zmf(u, [P(1) P(2)]);
        end
        if INV(i, j) == 0               % CHECK FOR NOT OPERATOR
            FS(i, :, j) = F;
        else
            FS(i, :, j) = 1 - F;
        end
    end
end
for i = 1:size(AMAT, 1)                            % i : Rule/Row Index
    for j = 1:size(FS, 3)                             % j : Antedent + Consequent Index
        if j <= size(AMAT, 2)
            AFS(i, :, j) = FS(i, :, j);      % AFS : Antecedent FS
        else
            CFS(i, :, j-size(AMAT, 2)) = FS(i, :, j);  % CFS : Consequent FS
        end
    end
end

for k = 1:size(input, 1)
    for i = 1:size(AMAT, 1)
        for j = 1:size(AMAT, 2)
            u = linspace(UD(j, 1), UD(j, 2), XA);
            in(j) = max(find(u <= input(k, j)));
            ACL(i, j) = AFS(i, in(j), j);               % CL :  Antecdent Clipping Level
            ACF(i, :, j) = min(ACL(i, j), AFS(i, :, j));  % CF : Antecedent Clipped Fuzzy Set
        end
    end

    % AND/OR OPERATORS
    for i = 1:size(CMAT,1) % No. of Rules
        if OP(i) == 1   %   T-NORM : AND Method
            switch gfs.TNorm
                case 'Minimum'                      % CCL : Consequent Clipping Level
                    CCL(i) = min(ACL(i, :));
                case 'Product'
                    CCL(i) = prod(ACL(i, :));
            end
        elseif OP(i) == 2   %   S-NORM : OR Method
            switch gfs.SNorm
                case 'Maximum'                      % CCL : Consequent Clipping Level
                    CCL(i) = max(ACL(i, :));
                case 'Probor'
                    CCL(i) = sum(ACL(i, :)) - prod(ACL(i, :));
            end
        end
    end

    for i = 1:size(CMAT,1) % No. of Rules
        for j = 1:size(CMAT,2) % No. of Consequents
            u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
            % IMPLICATION OPERATOR
            switch gfs.ImpliMethod
                case 'Minimum'
                    CCF(i, :, j) = min(CCL(i), CFS(i, :, j));     % CFS : Consequent Clipped FS - Minimum
                case 'Product'
                    CCF(i, :, j) = CCL(i)*CFS(i, :, j);     % CFS : Consequent clipped FS - Product
            end
        end
    end

    % ---- CONSEQUENT AGGREGATION SHADING ----
    for i = 1:size(CMAT,1) % No. of Rules
        for j = 1:size(CMAT,2) % No. of Consequents
            % AGGREGATION OPERATOR
            switch gfs.AggreMethod
                case 'Maximum'
                    AF(j, :) = max(CCF(:, :, j));   % AF : Aggregated Fuzzy - Consequent
                case 'Probor'
                    AF(j, :) = sum(CCF(:, :, j)) - prod(CCF(:, :, j));
            end
        end
    end

    % ---- DEFUZZIFICATION METHOD ----
    for j = 1:size(CMAT, 2)
        u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
        switch gfs.DeFuzzMethod
            case 'Centroid'
                DFUZ(j) = defuz(u, AF(j, :), 'centroid');
            case 'Bisector'
                DFUZ(j) = defuz(u, AF(j, :), 'bisector');
            case 'MoM'
                DFUZ(j) = defuz(u, AF(j, :), 'mom');
            case 'LoM'
                DFUZ(j) = defuz(u, AF(j, :), 'lom');
            case 'SoM'
                DFUZ(j) = defuz(u, AF(j, :), 'som');
        end
    end
    DFZ(k) = DFUZ;
end
% INTERVAL TYPE-2 FUZZY SYSTEM : APPROACH 1
% ===============================
function DFZ = it21gfs(input, gfs, XA)
NumIn = length(gfs.Input);
NumOut = length(gfs.Output);
NumRule = length(gfs.Rule);
for i = 1:NumIn
    UDI(i, :) = gfs.Input(1, i).UOD;
end
for j = 1:NumOut
    UDO(j, :) = gfs.Output(1, j).UOD;
end
UD = [UDI; UDO];

for i =1:NumRule
    AMAT(i, :) = gfs.Rule(1, i).antecedent;
    CMAT(i, :) = gfs.Rule(1, i).consequent;
    OP(i) = gfs.Rule(1, i).connection;
end
MFN = [AMAT CMAT];
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        if MFN(i, j) < 0, INV(i, j) = 1;
        else, INV(i, j) = 0;
        end
    end
end
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        U = UD(j, :);
        if j <= NumIn
            if MFN(i, j) ~= 0
                SL = gfs.Input(1, j).mf(1, abs(MFN(i, j))).LFSShape;                                        % SHAPES
                PL = gfs.Input(1, j).mf(1, abs(MFN(i, j))).LFSParams;             % PARAMETER
                SR = gfs.Input(1, j).mf(1, abs(MFN(i, j))).RFSShape;                                        % SHAPES
                PR = gfs.Input(1, j).mf(1, abs(MFN(i, j))).RFSParams;
            else
                SL = 'None'; SR = 'None';
            end
        elseif j > NumIn
            if MFN(i, j) ~= 0
                SL = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).LFSShape;
                PL = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).LFSParams;
                SR = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).RFSShape;
                PR = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).RFSParams;
            else
                SL = 'None'; SR = 'None';
            end
        end
        u = linspace(U(1), U(2), XA);
        switch SL
            case 'None'
                FSL = ones(1, length(u));
            case 'Gaussian'
                FSL = gaussmf(u, PL);
            case 'Trapezoidal'
                FSL = trapmf(u, PL);
            case 'Triangular'
                FSL = trimf(u, PL);
            case 'S-Shaped'
                FSL = smf(u, PL);
            case 'Z-Shaped'
                FSL = zmf(u, PL);
        end
        switch SR
            case 'None'
                FSR = ones(1, length(u));
            case 'Gaussian'
                FSR = gaussmf(u, PR);
            case 'Trapezoidal'
                FSR = trapmf(u, PR);
            case 'Triangular'
                FSR = trimf(u, PR);
            case 'S-Shaped'
                FSR = smf(u, PR);
            case 'Z-Shaped'
                FSR = zmf(u, PR);
        end
        % Conversion Left & Right FS into Upper & Lower MFs
        if INV(i, j) == 0   % INVERTOR OPERATOR
            UP = umf(FSL, FSR, u); LO = lmf(FSL, FSR, u);
        else
            UP = 1-lmf(FSL, FSR, u); LO = 1-umf(FSL, FSR, u);
        end
        ULFS(2*i-1:2*i, :, j) = [UP; LO];
    end
end
% plot(1:100, ULFS(:, :, 4))
for i = 1:size(AMAT, 1)                                % i : Rule/Row Index
    for j = 1:size(ULFS, 3)                             % j : Antedent & Consequent Index
        if j <= size(AMAT, 2)
            AUFS(i, :, j) = ULFS(2*i-1, :, j);      % AUFS : Antecedent Upper FS
            ALFS(i, :, j) = ULFS(2*i, :, j);           % ALFS : Antecedent Lower FS
        else
            CUFS(i, :, j-size(AMAT, 2)) = ULFS(2*i-1, :, j);  % CUFS : Consequent Upper FS
            CLFS(i, :, j-size(AMAT, 2)) = ULFS(2*i, :, j);      % CLFS : Consequent Lower FS
        end
    end
end

for i = 1:size(AMAT, 1)
    for j = 1:size(AMAT, 2)
        u = linspace(UD(j, 1), UD(j, 2), XA);
        in(j) = max(find(u <= input(j)));
        UC(i, j) = AUFS(i, in(j), j);               % UC :  Upper Clipping level of Antecdent
        LC(i, j) = ALFS(i, in(j), j);                % LC : Lower Clipping level of Antecdent

        UCF(i, :, j) = min(UC(i, j), AUFS(i, :, j));
        LCF(i, :, j) = min(LC(i, j), ALFS(i, :, j));
    end
end
% AND/OR OPERATORS
for i = 1:size(CMAT,1) % No. of Rules
    if OP(i) == 1   %   T-NORM : AND Method
        switch gfs.TNorm
            case 'Minimum'
                UCC(i) = min(UC(i, :));     % UCC : Upper Consequent Clipping level
                LCC(i) = min(LC(i, :));      % LCC : Lower Consequent Clipping level
            case 'Product'
                UCC(i) = prod(UC(i, :));
                LCC(i) = prod(LC(i, :));
        end
    elseif OP(i) == 2   %   T-NORM : OR Method
        switch gfs.SNorm
            case 'Maximum'
                UCC(i) = max(UC(i, :));
                LCC(i) = max(LC(i, :));
            case 'Probor'
                UCC(i) = sum(UC(i, :)) - prod(UC(i, :));
                LCC(i) = sum(LC(i, :)) - prod(LC(i, :));
        end
    end
end
for i = 1:size(CMAT,1) % No. of Rules
    for j = 1:size(CMAT,2) % No. of Consequents
        u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
        % IMPLICATION OPERATOR
        switch gfs.ImpliMethod
            case 'Minimum'
                UCFS(i, :, j) = min(UCC(i), CUFS(i, :, j));     % UCFS : Upper Consequent clipped FS
                LCFS(i, :, j) = min(LCC(i), CLFS(i, :, j));     % LCFS : Lower Consequent clipped FS
            case 'Product'
                UCFS(i, :, j) = UCC(i)*CUFS(i, :, j);     % UCFS : Upper Consequent clipped FS
                LCFS(i, :, j) = LCC(i)*CLFS(i, :, j);     % LCFS : Lower Consequent clipped FS
        end
    end
end
% ---- CONSEQUENT AGGREGATION SHADING ----
for j = 1:size(CMAT,2) % No. of Consequents
    % AGGREGATION OPERATOR
    switch gfs.AggreMethod
        case 'Maximum'
            UACF(j, :) = max(UCFS(:, :, j));   % ACF : Upper Aggregated Consequent Fuzzy
            LACF(j, :) = max(LCFS(:, :, j));     % LCF : Lower Aggregated Consequent Fuzzy
        case 'Probor'
            UACF(j, :) = sum(UCFS(:, :, j)) - prod(UCFS(:, :, j));
            LACF(j, :) = sum(LCFS(:, :, j)) - prod(LCFS(:, :, j));
    end
end
%  TYPE-REDUCTION METHOD
for j = 1:size(CMAT, 2)
    u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
    switch gfs.TRedMethod
        case 'Centroid'
            UDFUZ(j) = defuz(u, UACF(j, :), 'centroid');
            LDFUZ(j) = defuz(u, LACF(j, :), 'centroid');
        case 'Bisector'
            UDFUZ(j) = defuz(u, UACF(j, :), 'bisector');
            LDFUZ(j) = defuz(u, LACF(j, :), 'bisector');
        case 'MoM'
            UDFUZ(j) = defuz(u, UACF(j, :), 'mom');
            LDFUZ(j) = defuz(u, LACF(j, :), 'mom');
        case 'LoM'
            UDFUZ(j) = defuz(u, UACF(j, :), 'lom');
            LDFUZ(j) = defuz(u, LACF(j, :), 'lom');
        case 'SoM'
            UDFUZ(j) = defuz(u, UACF(j, :), 'som');
            LDFUZ(j) = defuz(u, LACF(j, :), 'som');
    end

    % ---- DEFUZZIFICATION METHOD ----
    switch gfs.DeFuzzMethod
        case 'Centroid'
            DFUZ(j) = (UDFUZ(j)+LDFUZ(j))/2;
        case 'Minimum'
            DFUZ(j) = min(UDFUZ(j), LDFUZ(j));
        case 'Maximum'
            DFUZ(j) = max(UDFUZ(j), LDFUZ(j));
    end
end

DFZ = DFUZ;

% INTERVAL TYPE-2 FUZZY SYSTEM : APPROACH 2
% ===============================
function DFZ = it22gfs(input, gfs, XA)
% assignin('base', 'sys', gfs)
NumIn = length(gfs.Input);
NumOut = length(gfs.Output);
NumRule = length(gfs.Rule);
for i = 1:NumIn
    UDI(i, :) = gfs.Input(1, i).UOD;
end
for j = 1:NumOut
    UDO(j, :) = gfs.Output(1, j).UOD;
end
UD = [UDI; UDO];

for i =1:NumRule
    AMAT(i, :) = gfs.Rule(1, i).antecedent;
    CMAT(i, :) = gfs.Rule(1, i).consequent;
    OP(i) = gfs.Rule(1, i).connection;
end
MFN = [AMAT CMAT];
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        if MFN(i, j) < 0, INV(i, j) = 1;
        else, INV(i, j) = 0;
        end
    end
end

for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        U = UD(j, :);
        if j <= NumIn
            if MFN(i, j) ~= 0
                SP = gfs.Input(1, j).mf(1, abs(MFN(i, j))).PMFShape;                                        % SHAPES
                PP = eval(gfs.Input(1, j).mf(1, abs(MFN(i, j))).PMFParams);             % PARAMETER
                FU = gfs.Input(1, j).mf(1, abs(MFN(i, j))).FOU;
            else
                SP = 'None';
            end
        elseif j > NumIn
            if MFN(i, j) ~= 0
                SP = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).PMFShape;
                PP = eval(gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).PMFParams);
                FU = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).FOU;
            else
                SP = 'None';
            end
        end
        u = linspace(U(1), U(2), XA);
        NM = 0.5;%(U(2)-U(1))/2;

        switch SP
            case 'None'
                FSL = ones(1, length(u));
                FSR = ones(1, length(u));
            case 'Gaussian'
                FSL = gaussmf(u, [PP(1) PP(2)-NM*FU]);
                FSR = gaussmf(u, [PP(1) PP(2)+NM*FU]);
            case 'Trapezoidal'
                FSL = trapmf(u, PP-NM*FU);
                FSR = trapmf(u, PP+NM*FU);
            case 'Triangular'
                FSL = trimf(u, PP-NM*FU);
                FSR = trimf(u, PP+NM*FU);
            case 'S-Shaped'
                FSL = smf(u, PP-NM*FU);
                FSR = smf(u, PP+NM*FU);
            case 'Z-Shaped'
                FSL = zmf(u, PP-NM*FU);
                FSR = zmf(u, PP+NM*FU);
        end
        % Conversion Left & Right FS into Upper & Lower MFs
        if INV(i, j) == 0   % INVERTOR OPERATOR
            UP = umf(FSL, FSR, u); LO = lmf(FSL, FSR, u);
        else
            UP = 1-lmf(FSL, FSR, u); LO = 1-umf(FSL, FSR, u);
        end
        ULFS(2*i-1:2*i, :, j) = [UP; LO];
    end
end
% plot(1:100, ULFS(:, :, 4))
for i = 1:size(AMAT, 1)                                % i : Rule/Row Index
    for j = 1:size(ULFS, 3)                             % j : Antedent & Consequent Index
        if j <= size(AMAT, 2)
            AUFS(i, :, j) = ULFS(2*i-1, :, j);      % AUFS : Antecedent Upper FS
            ALFS(i, :, j) = ULFS(2*i, :, j);           % ALFS : Antecedent Lower FS
        else
            CUFS(i, :, j-size(AMAT, 2)) = ULFS(2*i-1, :, j);  % CUFS : Consequent Upper FS
            CLFS(i, :, j-size(AMAT, 2)) = ULFS(2*i, :, j);      % CLFS : Consequent Lower FS
        end
    end
end
for k = 1:size(input, 1)
    for i = 1:size(AMAT, 1)
        for j = 1:size(AMAT, 2)
            u = linspace(UD(j, 1), UD(j, 2), XA);
            in(j) = max(find(u <= input(k, j)));
            UC(i, j) = AUFS(i, in(j), j);               % UC :  Upper Clipping level of Antecdent
            LC(i, j) = ALFS(i, in(j), j);                % LC : Lower Clipping level of Antecdent

            UCF(i, :, j) = min(UC(i, j), AUFS(i, :, j));
            LCF(i, :, j) = min(LC(i, j), ALFS(i, :, j));
        end
    end
    % AND/OR OPERATORS
    for i = 1:size(CMAT,1) % No. of Rules
        if OP(i) == 1   %   T-NORM : AND Method
            switch gfs.TNorm
                case 'Minimum'
                    UCC(i) = min(UC(i, :));     % UCC : Upper Consequent Clipping level
                    LCC(i) = min(LC(i, :));      % LCC : Lower Consequent Clipping level
                case 'Product'
                    UCC(i) = prod(UC(i, :));
                    LCC(i) = prod(LC(i, :));
            end
        elseif OP(i) == 2   %   T-NORM : OR Method
            switch gfs.SNorm
                case 'Maximum'
                    UCC(i) = max(UC(i, :));
                    LCC(i) = max(LC(i, :));
                case 'Probor'
                    UCC(i) = sum(UC(i, :)) - prod(UC(i, :));
                    LCC(i) = sum(LC(i, :)) - prod(LC(i, :));
            end
        end
    end
    for i = 1:size(CMAT,1) % No. of Rules
        for j = 1:size(CMAT,2) % No. of Consequents
            u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
            % IMPLICATION OPERATOR
            switch gfs.ImpliMethod
                case 'Minimum'
                    UCFS(i, :, j) = min(UCC(i), CUFS(i, :, j));     % UCFS : Upper Consequent clipped FS
                    LCFS(i, :, j) = min(LCC(i), CLFS(i, :, j));     % LCFS : Lower Consequent clipped FS
                case 'Product'
                    UCFS(i, :, j) = UCC(i)*CUFS(i, :, j);     % UCFS : Upper Consequent clipped FS
                    LCFS(i, :, j) = LCC(i)*CLFS(i, :, j);     % LCFS : Lower Consequent clipped FS
            end
        end
    end
    % ---- CONSEQUENT AGGREGATION SHADING ----
    for j = 1:size(CMAT,2) % No. of Consequents
        % AGGREGATION OPERATOR
        switch gfs.AggreMethod
            case 'Maximum'
                UACF(j, :) = max(UCFS(:, :, j));   % ACF : Upper Aggregated Consequent Fuzzy
                LACF(j, :) = max(LCFS(:, :, j));     % LCF : Lower Aggregated Consequent Fuzzy
            case 'Probor'
                UACF(j, :) = sum(UCFS(:, :, j)) - prod(UCFS(:, :, j));
                LACF(j, :) = sum(LCFS(:, :, j)) - prod(LCFS(:, :, j));
        end
    end
    %  TYPE-REDUCTION METHOD
    for j = 1:size(CMAT, 2)
        u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);

        switch gfs.TRedMethod
            case 'Centroid'
                UDFUZ(j) = defuz(u, UACF(j, :), 'centroid');
                LDFUZ(j) = defuz(u, LACF(j, :), 'centroid');
            case 'Bisector'
                UDFUZ(j) = defuz(u, UACF(j, :), 'bisector');
                LDFUZ(j) = defuz(u, LACF(j, :), 'bisector');
            case 'MoM'
                UDFUZ(j) = defuz(u, UACF(j, :), 'mom');
                LDFUZ(j) = defuz(u, LACF(j, :), 'mom');
            case 'LoM'
                UDFUZ(j) = defuz(u, UACF(j, :), 'lom');
                LDFUZ(j) = defuz(u, LACF(j, :), 'lom');
            case 'SoM'
                UDFUZ(j) = defuz(u, UACF(j, :), 'som');
                LDFUZ(j) = defuz(u, LACF(j, :), 'som');
        end

        % ---- DEFUZZIFICATION METHOD ----
        switch gfs.DeFuzzMethod
            case 'Centroid'
                DFUZ(j) = (UDFUZ(j)+LDFUZ(j))/2;
            case 'Minimum'
                DFUZ(j) = min(UDFUZ(j), LDFUZ(j));
            case 'Maximum'
                DFUZ(j) = max(UDFUZ(j), LDFUZ(j));
        end
    end

    DFZ(k) = DFUZ;
end
% INTERVAL TYPE-2 FUZZY SYSTEM : APPROACH 3
% ===============================
function DFZ = it23gfs(input, gfs, XA)
NumIn = length(gfs.Input);
NumOut = length(gfs.Output);
NumRule = length(gfs.Rule);
for i = 1:NumIn
    UDI(i, :) = gfs.Input(1, i).UOD;
end
for j = 1:NumOut
    UDO(j, :) = gfs.Output(1, j).UOD;
end
UD = [UDI; UDO];

for i =1:NumRule
    AMAT(i, :) = gfs.Rule(1, i).antecedent;
    CMAT(i, :) = gfs.Rule(1, i).consequent;
    OP(i) = gfs.Rule(1, i).connection;
end
MFN = [AMAT CMAT];
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        if MFN(i, j) < 0, INV(i, j) = 1;
        else, INV(i, j) = 0;
        end
    end
end
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        U = UD(j, :);
        if j <= NumIn
            if MFN(i, j) ~= 0
                SU = gfs.Input(1, j).mf(1, abs(MFN(i, j))).UMFShape;                                        % SHAPES
                PU = gfs.Input(1, j).mf(1, abs(MFN(i, j))).UMFParams;             % PARAMETER
                SL = gfs.Input(1, j).mf(1, abs(MFN(i, j))).LMFShape;                                        % SHAPES
                PL = gfs.Input(1, j).mf(1, abs(MFN(i, j))).LMFParams;             % PARAMETER
            else
                SU = 'None'; SL = 'None';
            end
        elseif j > NumIn
            if MFN(i, j) ~= 0
                SU = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).UMFShape;
                PU = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).UMFParams;
                SL = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).LMFShape;
                PL = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).LMFParams;
            else
                SU = 'None'; SL = 'None';
            end
        end
        u = linspace(U(1), U(2), XA);
        switch SU
            case 'None'
                FSU = ones(1, length(u));
            case 'Gaussian'
                FSU = gaussmf(u, PU);
            case 'Trapezoidal'
                FSU = trapmf(u, PU);
            case 'Triangular'
                FSU = trimf(u, PU);
            case 'S-Shaped'
                FSU = smf(u, PU);
            case 'Z-Shaped'
                FSU = zmf(u, PU);
        end
        switch SL
            case 'None'
                FSL = ones(1, length(u));
            case 'Gaussian'
                FSL = PL(3)*gaussmf(u, PL(1:2));
            case 'Trapezoidal'
                FSL = PL(5)*trapmf(u, PL(1:4));
            case 'Triangular'
                FSL = PL(4)*trimf(u, PL(1:3));
            case 'S-Shaped'
                FSL = PL(3)*smf(u, PL(1:2));
            case 'Z-Shaped'
                FSL = PL(3)*zmf(u, PL(1:2));
        end
        % Conversion Left & Right FS into Upper & Lower MFs
        if INV(i, j) == 0   % INVERTOR OPERATOR
            UP = FSU; LO = FSL;
        else
            UP = 1-FSU; LO = 1-FSL;
        end
        ULFS(2*i-1:2*i, :, j) = [UP; LO];
    end
end
% plot(1:100, ULFS(:, :, 4))
for i = 1:size(AMAT, 1)                                % i : Rule/Row Index
    for j = 1:size(ULFS, 3)                             % j : Antedent & Consequent Index
        if j <= size(AMAT, 2)
            AUFS(i, :, j) = ULFS(2*i-1, :, j);      % AUFS : Antecedent Upper FS
            ALFS(i, :, j) = ULFS(2*i, :, j);           % ALFS : Antecedent Lower FS
        else
            CUFS(i, :, j-size(AMAT, 2)) = ULFS(2*i-1, :, j);  % CUFS : Consequent Upper FS
            CLFS(i, :, j-size(AMAT, 2)) = ULFS(2*i, :, j);      % CLFS : Consequent Lower FS
        end
    end
end

for i = 1:size(AMAT, 1)
    for j = 1:size(AMAT, 2)
        u = linspace(UD(j, 1), UD(j, 2), XA);
        in(j) = max(find(u <= input(j)));
        UC(i, j) = AUFS(i, in(j), j);               % UC :  Upper Clipping level of Antecdent
        LC(i, j) = ALFS(i, in(j), j);                % LC : Lower Clipping level of Antecdent

        UCF(i, :, j) = min(UC(i, j), AUFS(i, :, j));
        LCF(i, :, j) = min(LC(i, j), ALFS(i, :, j));
    end
end
% AND/OR OPERATORS
for i = 1:size(CMAT,1) % No. of Rules
    if OP(i) == 1   %   T-NORM : AND Method
        switch gfs.TNorm
            case 'Minimum'
                UCC(i) = min(UC(i, :));     % UCC : Upper Consequent Clipping level
                LCC(i) = min(LC(i, :));      % LCC : Lower Consequent Clipping level
            case 'Product'
                UCC(i) = prod(UC(i, :));
                LCC(i) = prod(LC(i, :));
        end
    elseif OP(i) == 2   %   T-NORM : OR Method
        switch gfs.SNorm
            case 'Maximum'
                UCC(i) = max(UC(i, :));
                LCC(i) = max(LC(i, :));
            case 'Probor'
                UCC(i) = sum(UC(i, :)) - prod(UC(i, :));
                LCC(i) = sum(LC(i, :)) - prod(LC(i, :));
        end
    end
end
for i = 1:size(CMAT,1) % No. of Rules
    for j = 1:size(CMAT,2) % No. of Consequents
        u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
        % IMPLICATION OPERATOR
        switch gfs.ImpliMethod
            case 'Minimum'
                UCFS(i, :, j) = min(UCC(i), CUFS(i, :, j));     % UCFS : Upper Consequent clipped FS
                LCFS(i, :, j) = min(LCC(i), CLFS(i, :, j));     % LCFS : Lower Consequent clipped FS
            case 'Product'
                UCFS(i, :, j) = UCC(i)*CUFS(i, :, j);     % UCFS : Upper Consequent clipped FS
                LCFS(i, :, j) = LCC(i)*CLFS(i, :, j);     % LCFS : Lower Consequent clipped FS
        end
    end
end
% ---- CONSEQUENT AGGREGATION SHADING ----
for j = 1:size(CMAT,2) % No. of Consequents
    % AGGREGATION OPERATOR
    switch gfs.AggreMethod
        case 'Maximum'
            UACF(j, :) = max(UCFS(:, :, j));   % ACF : Upper Aggregated Consequent Fuzzy
            LACF(j, :) = max(LCFS(:, :, j));     % LCF : Lower Aggregated Consequent Fuzzy
        case 'Probor'
            UACF(j, :) = sum(UCFS(:, :, j)) - prod(UCFS(:, :, j));
            LACF(j, :) = sum(LCFS(:, :, j)) - prod(LCFS(:, :, j));
    end
end
%  TYPE-REDUCTION METHOD
for j = 1:size(CMAT, 2)
    u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
    switch gfs.TRedMethod
        case 'Centroid'
            UDFUZ(j) = defuz(u, UACF(j, :), 'centroid');
            LDFUZ(j) = defuz(u, LACF(j, :), 'centroid');
        case 'Bisector'
            UDFUZ(j) = defuz(u, UACF(j, :), 'bisector');
            LDFUZ(j) = defuz(u, LACF(j, :), 'bisector');
        case 'MoM'
            UDFUZ(j) = defuz(u, UACF(j, :), 'mom');
            LDFUZ(j) = defuz(u, LACF(j, :), 'mom');
        case 'LoM'
            UDFUZ(j) = defuz(u, UACF(j, :), 'lom');
            LDFUZ(j) = defuz(u, LACF(j, :), 'lom');
        case 'SoM'
            UDFUZ(j) = defuz(u, UACF(j, :), 'som');
            LDFUZ(j) = defuz(u, LACF(j, :), 'som');
    end

    % ---- DEFUZZIFICATION METHOD ----
    switch gfs.DeFuzzMethod
        case 'Centroid'
            DFUZ(j) = (UDFUZ(j)+LDFUZ(j))/2;
        case 'Minimum'
            DFUZ(j) = min(UDFUZ(j), LDFUZ(j));
        case 'Maximum'
            DFUZ(j) = max(UDFUZ(j), LDFUZ(j));
    end
end

DFZ = DFUZ;

% TRIANGULAR TYPE-2 FUZZY SYSTEM : APPROACH 1
% ===============================
function DFZ = tt21gfs(input, gfs, XA)
NumIn = length(gfs.Input);
NumOut = length(gfs.Output);
NumRule = length(gfs.Rule);
for i = 1:NumIn
    UDI(i, :) = gfs.Input(1, i).UOD;
end
for j = 1:NumOut
    UDO(j, :) = gfs.Output(1, j).UOD;
end
UD = [UDI; UDO];

for i =1:NumRule
    AMAT(i, :) = gfs.Rule(1, i).antecedent;
    CMAT(i, :) = gfs.Rule(1, i).consequent;
    OP(i) = gfs.Rule(1, i).connection;
end
MFN = [AMAT CMAT];
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        if MFN(i, j) < 0, INV(i, j) = 1;
        else, INV(i, j) = 0;
        end
    end
end
for i = 1:size(MFN, 1)          % No. of Rules or Rows
    for j = 1:size(MFN, 2)      % No. of Inputs+Outputs or Columns
        U = UD(j, :);
        if j <= NumIn
            if MFN(i, j) ~= 0
                SL = gfs.Input(1, j).mf(1, abs(MFN(i, j))).LFSShape;                                        % SHAPES
                PL = (gfs.Input(1, j).mf(1, abs(MFN(i, j))).LFSParams);             % PARAMETER

                SR = gfs.Input(1, j).mf(1, abs(MFN(i, j))).RFSShape;                                        % SHAPES
                PR = (gfs.Input(1, j).mf(1, abs(MFN(i, j))).RFSParams);             % PARAMETER

                SP = gfs.Input(1, j).mf(1, abs(MFN(i, j))).PFSShape;                                        % SHAPES
                PP = (gfs.Input(1, j).mf(1, abs(MFN(i, j))).PFSParams);             % PARAMETER
            else
                SL = 'None'; SR = 'None'; SP = 'None';
            end
        elseif j > NumIn
            if MFN(i, j) ~= 0
                SL = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).LFSShape;
                PL = (gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).LFSParams);

                SR = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).RFSShape;
                PR = (gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).RFSParams);

                SP = gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).PFSShape;
                PP = (gfs.Output(1, j-NumIn).mf(1, abs(MFN(i, j))).PFSParams);
            else
                SL = 'None'; SR = 'None'; SP = 'None';
            end
        end
        u = linspace(U(1), U(2), XA);
        switch SL
            case 'None'
                FSL = ones(1, length(u));
            case 'Gaussian'
                FSL = gaussmf(u, PL);
            case 'Trapezoidal'
                FSL = trapmf(u, PL);
            case 'Triangular'
                FSL = trimf(u, PL);
            case 'S-Shaped'
                FSL = smf(u, PL);
            case 'Z-Shaped'
                FSL = zmf(u, PL);
        end
        switch SR
            case 'None'
                FSR = ones(1, length(u));
            case 'Gaussian'
                FSR = gaussmf(u, PR);
            case 'Trapezoidal'
                FSR = trapmf(u, PR);
            case 'Triangular'
                FSR = trimf(u, PR);
            case 'S-Shaped'
                FSR = smf(u, PR);
            case 'Z-Shaped'
                FSR = zmf(u, PR);
        end
        switch SP
            case 'None'
                FSP = ones(1, length(u));
            case 'Gaussian'
                FSP = gaussmf(u, PP);
            case 'Trapezoidal'
                FSP = trapmf(u, PP);
            case 'Triangular'
                FSP = trimf(u, PP);
            case 'S-Shaped'
                FSP = smf(u, PP);
            case 'Z-Shaped'
                FSP = zmf(u, PP);
        end
        % Conversion Left & Right FS into Upper & Lower MFs
        if INV(i, j) == 0   % INVERTOR OPERATOR
            UP = umf(FSL, FSR, u); LO = lmf(FSL, FSR, u);
        else
            UP = 1-lmf(FSL, FSR, u); LO = 1-umf(FSL, FSR, u);
        end
        ULFS(2*i-1:2*i+1, :, j) = [UP; FSP; LO];
    end
end
% plot(1:100, ULFS(:, :, 4))
for i = 1:size(AMAT, 1)                                % i : Rule/Row Index
    for j = 1:size(ULFS, 3)                             % j : Antedent & Consequent Index
        if j <= size(AMAT, 2)
            AUFS(i, :, j) = ULFS(2*i-1, :, j);      % AUFS : Antecedent Upper FS
            APFS(i, :, j) = ULFS(2*i, :, j);           % APFS : Antecedent Principal FS
            ALFS(i, :, j) = ULFS(2*i+1, :, j);           % ALFS : Antecedent Lower FS
        else
            CUFS(i, :, j-size(AMAT, 2)) = ULFS(2*i-1, :, j);  % CUFS : Consequent Upper FS
            CPFS(i, :, j-size(AMAT, 2)) = ULFS(2*i, :, j);      % CPFS : Consequent Principal FS
            CLFS(i, :, j-size(AMAT, 2)) = ULFS(2*i+1, :, j);      % CLFS : Consequent Lower FS
        end
    end
end

for k = 1:size(input, 1)
    for i = 1:size(AMAT, 1)
        for j = 1:size(AMAT, 2)
            u = linspace(UD(j, 1), UD(j, 2), XA);
            in(j) = max(find(u <= input(k, j)));
            UC(i, j) = AUFS(i, in(j), j);               % UC :  Upper Clipping level of Antecdent
            PC(i, j) = APFS(i, in(j), j);                % LP : Principal Clipping level of Antecdent
            LC(i, j) = ALFS(i, in(j), j);                % LC : Lower Clipping level of Antecdent

            UCF(i, :, j) = min(UC(i, j), AUFS(i, :, j));
            PCF(i, :, j) = min(PC(i, j), APFS(i, :, j));
            LCF(i, :, j) = min(LC(i, j), ALFS(i, :, j));
        end
    end
    % AND/OR OPERATORS
    for i = 1:size(CMAT,1) % No. of Rules
        if OP(i) == 1   %   T-NORM : AND Method
            switch gfs.TNorm
                case 'Minimum'
                    UCC(i) = min(UC(i, :));     % UCC : Upper Consequent Clipping level
                    PCC(i) = min(PC(i, :));      % PCC : Principal Consequent Clipping level
                    LCC(i) = min(LC(i, :));      % LCC : Lower Consequent Clipping level
                case 'Product'
                    UCC(i) = prod(UC(i, :));
                    PCC(i) = prod(PC(i, :));
                    LCC(i) = prod(LC(i, :));
            end
        elseif OP(i) == 2   %   T-NORM : OR Method
            switch gfs.SNorm
                case 'Maximum'
                    UCC(i) = max(UC(i, :));
                    PCC(i) = max(PC(i, :));
                    LCC(i) = max(LC(i, :));
                case 'Probor'
                    UCC(i) = sum(UC(i, :)) - prod(UC(i, :));
                    PCC(i) = sum(PC(i, :)) - prod(PC(i, :));
                    LCC(i) = sum(LC(i, :)) - prod(LC(i, :));
            end
        end
    end
    for i = 1:size(CMAT,1) % No. of Rules
        for j = 1:size(CMAT,2) % No. of Consequents
            u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
            % IMPLICATION OPERATOR
            switch gfs.ImpliMethod
                case 'Minimum'
                    UCFS(i, :, j) = min(UCC(i), CUFS(i, :, j));     % UCFS : Upper Consequent clipped FS
                    PCFS(i, :, j) = min(PCC(i), CPFS(i, :, j));     % PCFS : Principal Consequent clipped FS
                    LCFS(i, :, j) = min(LCC(i), CLFS(i, :, j));     % LCFS : Lower Consequent clipped FS
                case 'Product'
                    UCFS(i, :, j) = UCC(i)*CUFS(i, :, j);     % UCFS : Upper Consequent clipped FS
                    PCFS(i, :, j) = PCC(i)*CPFS(i, :, j);     % LCFS : Lower Consequent clipped FS
                    LCFS(i, :, j) = LCC(i)*CLFS(i, :, j);     % LCFS : Lower Consequent clipped FS
            end
        end
    end
    % ---- CONSEQUENT AGGREGATION SHADING ----
    for j = 1:size(CMAT,2) % No. of Consequents
        % AGGREGATION OPERATOR
        switch gfs.AggreMethod
            case 'Maximum'
                UACF(j, :) = max(UCFS(:, :, j));   % ACF : Upper Aggregated Consequent Fuzzy
                PACF(j, :) = max(PCFS(:, :, j));     % PCF : Principal Aggregated Consequent Fuzzy
                LACF(j, :) = max(LCFS(:, :, j));     % LCF : Lower Aggregated Consequent Fuzzy
            case 'Probor'
                UACF(j, :) = sum(UCFS(:, :, j)) - prod(UCFS(:, :, j));
                PACF(j, :) = sum(PCFS(:, :, j)) - prod(PCFS(:, :, j));
                LACF(j, :) = sum(LCFS(:, :, j)) - prod(LCFS(:, :, j));
        end
    end
    %  TYPE-REDUCTION METHOD
    for j = 1:size(CMAT, 2)
        u = linspace(UD(j+NumIn, 1), UD(j+NumIn, 2), XA);
        switch gfs.TRedMethod
            case 'Centroid'
                UDFUZ(j) = defuz(u, UACF(j, :), 'centroid');
                PDFUZ(j) = defuz(u, PACF(j, :), 'centroid');
                LDFUZ(j) = defuz(u, LACF(j, :), 'centroid');
            case 'Bisector'
                UDFUZ(j) = defuz(u, UACF(j, :), 'bisector');
                PDFUZ(j) = defuz(u, PACF(j, :), 'bisector');
                LDFUZ(j) = defuz(u, LACF(j, :), 'bisector');
            case 'MoM'
                UDFUZ(j) = defuz(u, UACF(j, :), 'mom');
                PDFUZ(j) = defuz(u, PACF(j, :), 'mom');
                LDFUZ(j) = defuz(u, LACF(j, :), 'mom');
            case 'LoM'
                UDFUZ(j) = defuz(u, UACF(j, :), 'lom');
                PDFUZ(j) = defuz(u, PACF(j, :), 'lom');
                LDFUZ(j) = defuz(u, LACF(j, :), 'lom');
            case 'SoM'
                UDFUZ(j) = defuz(u, UACF(j, :), 'som');
                PDFUZ(j) = defuz(u, PACF(j, :), 'som');
                LDFUZ(j) = defuz(u, LACF(j, :), 'som');
        end

        % ---- DEFUZZIFICATION METHOD ----
        switch gfs.DeFuzzMethod
            case 'Centroid'
                OM = [UDFUZ(j) PDFUZ(j) LDFUZ(j)];
                DFUZ(j) = defuz(u, trimf(u, sort(OM, 'ascend')), 'centroid');
            case 'Minimum'
                DFUZ(j) = min(UDFUZ(j), PDFUZ(j), LDFUZ(j));
            case 'Maximum'
                DFUZ(j) = max(UDFUZ(j), PDFUZ(j), LDFUZ(j));
        end
    end
    DFZ(k,:) = DFUZ;
end